; ****************************************************************************
; * FREQUENCY COUNTER WITH PROGRAMABLE IF OFFSET	                     *
; *  Version d3                                                              *
; *  May 11, 2006                                                            *
; ****************************************************************************
; Description:
;
; Sources of inspiration for this project- 
;
; The NJ-QRP club siggen_3a DDS project from which much of 
; the LCD, delay and EEPROM routines were adapted.
;
; A special mention and thanks to the siggen_3a authors;
; 	
;	Curt Preuss	WB2V	Author of the original siggen code
;	Craig Johnson 	AA0ZZ}	Joint Authors of the siggen_3a code
;	Bruce Stough	AA0ED}	developed from Curt's original
;	George Heron	N2APB	For publishing the code	
;
;
; Peter Halicky OM3CPH for the concept of using an internal
; register for MSB storage, also for ideas on how to write
; the gate routine.
;
; EI9GQ for the external gate circuitry
;
; Thanks to everyone!
;
; This software is provided free in the spirit of Ham Radio for all to use.
; Do not exploit this gesture
;
; Ron Taylor G4GXO
; 
; Version 5a	First release
;	  6a	Zero Supression	
;	  7a	Modified software for new gate configuration
;	  8a	Frequency Multiplier Option added
;	  9c	Published (internet) working version
;	  d1	Tidy up to remove compilation warnings	
;	  d2	Macro section, initial EEPROM constants, message status added
;         d3    Bug in Prescaler evaluation, sometimes qrg 2,56KHz to high, solved by
;               Andreas, DL5MGD
;
;                         
; Target Controller -      PIC16F84
;                          __________
;  Clock       -------RA2 |1       18| RA1---------Add/Sub IF
;  Gate   PB_Down-----RA3 |2       17| RA0---------Use IF Offset? (High Yes, Low No)
;  RF I/P PB_Up-------RA4 |3       16| OSC1--------XTAL
;     	    --------!MCLR |4       15| OSC2--------XTAL
;     Ground----------Vss |5       14| VDD---------+5 V
;     		  ----RB0 |6       13| RB7---------LCD 14
;     LCD_rs----------RB1 |7       12| RB6---------LCD 13
;     LCD_rw----------RB2 |8       11| RB5---------LCD 12
;     LCD_e-----------RB3 |9       10| RB4---------LCD 11
;                          ----------

;
; ****************************************************************************
; * Device type and options.                                                 *
; ****************************************************************************
;
        processor       PIC16F84
        radix           dec
;
; ****************************************************************************
; * Configuration fuse information:                                          *
; ****************************************************************************

_CP_ON                       EQU     H'000F'
_CP_OFF                      EQU     H'3FFF'
_PWRTE_ON                    EQU     H'3FF7'
_PWRTE_OFF                   EQU     H'3FFF'
_WDT_ON                      EQU     H'3FFF'
_WDT_OFF                     EQU     H'3FFB'
_LP_OSC                      EQU     H'3FFC'
_XT_OSC                      EQU     H'3FFD'
_HS_OSC                      EQU     H'3FFE'
_RC_OSC                      EQU     H'3FFF'
;
        __config        _CP_OFF & _PWRTE_ON & _XT_OSC & _WDT_OFF
;
;
; NOTE WDT IS OFF!
; ================
;
; ============================================================================
; Setup initial IF Offset and Multiplier Constant      
; ============================================================================ 
; 
; Setup initial IF Offset as a Hex Value, this can be changed by user once
; counter is operational. Standard IF of 9MHz is used for convenience.
; Because resolution is 10Hz offset is 900,000 = 0DBBA0
;	
;
        ORG     0x2100
        DATA    0xA0	; LSB
        DATA    0xBB
        DATA    0x0D
        DATA    0x00	; MSB
;
; Set initial Multiplier value to 1
;
	DATA	0x01	; Multiplier Factor
;
; Set display state to perm (1)
;
	DATA	0x01	; Display on 
 
;
;       Clear unused EEPROM bytes.
;
        DATA    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
        DATA    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
;
;
; ****************************************************************************
;                     Port and EEPROM Constants                             
; ****************************************************************************
;
PortA   equ     0x05
PortB   equ     0x06
TRISA   equ     0x05
TRISB   equ     0x06
EEdata  equ     0x08
EEadr   equ     0x09
WREN    equ     0x02
WR      equ     0x01
RD      equ     0x00
;
; ****************************************************************************
; * RAM page independent file registers:                                     *
; ****************************************************************************
;
INDF    EQU     0x00
TMR0	EQU	0x01
OPTN	EQU	0X01
PCL     EQU     0x02
STATUS  EQU     0x03
FSR     EQU     0x04
PCLATH  EQU     0x0A
INTCON  EQU     0x0B

;
; *****************************************************************************
; * Bit numbers for the STATUS file register:                                 *
; *****************************************************************************
;
B_RP1	EQU	6
B_RP0   EQU     5
B_NTO   EQU     4
B_NPD   EQU     3
B_Z     EQU     2
B_DC    EQU     1
B_C     EQU     0
;
; ****************************************************************************
; * Assign names to IO pins.                                                 *
; ****************************************************************************
;
;   A register bits:
;

IF_En	equ	0x00		; 1=Enable IF Offset, 0=Direct Freq Measurement
IF_AS	equ	0x01		; 1=Add, 0=Subtract
Clock	equ	0x02		; Count prescaler contents
PB_Down	equ	0x03		; Down Step Button
PB_Up	equ	0x04		; Up Step Button

;
;   B register bits:
;


LCD_rs  equ     0x01              ; 0=instruction, 1=data
LCD_rw  equ     0x02              ; 0=write, 1=read
LCD_e   equ     0x03              ; 0=disable, 1=enable
	

; ****************************************************************************
; *           Allocate variables in general purpose register space           *
; ****************************************************************************
;
        CBLOCK  0x0c              ; Start Data Block
;
        freq_0                    ; Display frequency (binary) 
          freq_1                  ;  (4 bytes) 
          freq_2
          freq_3
	xfreq_0			  ; Temporary storage for
	  xfreq_1		  ; multiplier routine
	  xfreq_2
	  xfreq_3         
        BCD_0                     ; Display frequency (BCD) 
          BCD_1                   ;  (5 bytes)
          BCD_2
          BCD_3
          BCD_4     
        BCD_count                 ; Used in bin2BCD routine
        BCD_temp                  ;   "
        bit_count                 ; Used in serial inteface
        LCD_char                  ; Character being sent to the LCD
        LCD_read                  ; Character read from the LCD
        timer1                    ; Used in delay routines
        timer2                    ;   "
        count                     ; loop counter  (gets reused)
        rs_value                  ; The LCD rs line flag value
	rxbyte			  ; Used to receive data from PIC1
	delay1			  ; Used in Gate delay
	delay2			  ; Used in Gate delay
	IF_0		  	  ; Used in IF Offset Routine
	IF_1		  	  ;
	IF_2		  	  ;
	IF_3		  
	IF_Range		  ; Used to get Prog Switch Value
	IF_Step			  ; Pointer for IF Table 
	IF_Step_3		  ; MSB IF Programming Step Size
	IF_Step_2		  
	IF_Step_1
	IF_Step_0		  ; LSB IF Programming Step Size
	temp			  ; General temporary store
	zeroflag		  ; Used in zero supression code
	msgflag			  ; Check for new message
	msgflag1		  ; Used for programming message status
	msg_state		  ; Message perm/timed state
	x_freq			  ; Frequency Multiplier factor
	x_BCD			  ; BCD Value of x_freq
	msg_timer		  ; Delay counter for message display
	msgflag_old		  ; Histroic store for msgflag change check
;
        ENDC                      ; End of Data Block

; ****************************************************************************

; Macro Area

; Bank Select Macros to simplify bank changes

Bank0	MACRO
	bcf	STATUS,B_RP0	; Select Bank 0
	bcf	STATUS,B_RP1	;
	ENDM

Bank1	MACRO
	bsf	STATUS,B_RP0	; Select Bank 1
	bcf	STATUS,B_RP1	;
	ENDM

; 
; ****************************************************************************
; * The 16F84 resets to 0x00.                                                * 
; * The Interrupt vector is at 0x04. (Unused)                                *
; ****************************************************************************             
;
        ORG     0x0000                
reset_entry
        goto    start             ;
; Table of decades with x10 offset 100mSec gate)

Step_Table
        addwf   PCL,f             ; 
        retlw   0x01              ; 10 Hz 	LSB
        retlw   0x00              ; 
        retlw   0x00              ;
        retlw   0x00              ;		MSB
        retlw   0x0A              ; 100 Hz	LSB
        retlw   0x00              ; 
        retlw   0x00              ;
        retlw   0x00              ;		MSB
        retlw   0x64              ; 1000 Hz	LSB
        retlw   0x00              ; 
        retlw   0x00              ;
        retlw   0x00              ;		MSB
        retlw   0xE8              ; 10 kHz	LSB
        retlw   0x03              ; 
        retlw   0x00              ;
        retlw   0x00              ;		MSB
        retlw   0x10              ; 100 kHz	LSB
        retlw   0x27              ; 
        retlw   0x00              ;
        retlw   0x00              ;		MSB
	retlw   0xA0              ; 1 MHz 	LSB
        retlw   0x86              ; 
        retlw   0x01              ;
        retlw   0x00              ;		MSB
        retlw   0x40              ; 10 MHz	LSB
        retlw   0x42              ; 
        retlw   0x0F              ;
        retlw   0x00              ;		MSB
        retlw   0x80              ; 100 MHz	LSB
        retlw   0x96              ; 
        retlw   0x98              ;
        retlw   0x00              ;		MSB
        
;
; *****************************************************************************
; *                                                                           *
; * Purpose:  This is the start of the program. Ports and LCD are initialised *
; *           before entering the main loop.                                  *
; *                     		                                      *
; *                                                                           *
; *****************************************************************************
;
start


  	clrf    INTCON            ; No interrupts for now
	Bank1			  ; Switch to bank 1
        bsf     0x01,7            ; Disable weak pullups
        movlw   b'00011111'       ; Set port A to all inputs
        movwf   TRISA             ;
	movlw	b'00000000'	  ;
        movwf   TRISB             ; Set port B 0..7 to outputs
	movlw	b'10110111' 	  ; Load Option register with prescalar and counter
	movwf	OPTN		  ; Set OPTION Reg for 1 to 256, falling edge	
	Bank0			  ; Switch back to bank 0
	call	read_EEifs	  ; Load IF Frequency, mult factor and message status from EEPROM
        call    init_LCD          ; Initialize the LCD
				  ; Test for Power On IF Programming Mode
	movlw	0x08		  ; Set up counter for Power On IF Programming Mode test
	movwf	count
Mode	
	clrf	msgflag
	btfsc	PortA, PB_Down	  ; Is Push Button 1 pressed?	(Normal 1, Operated 0)
	goto	main		  ; No
	btfss	PortA, PB_Up	  ; Is Push Button 2 pressed?	(Normal 0, Operated 1)
	goto	main		  ; No	
	call 	wait_128ms	  ; Loop delay of about 1 Second
	decfsz	count,f
	goto	Mode	
	call	IF_Set		  ; Configure IF Offset Frequency
	call	mult_set	  ; Set frequency multiplier
	call	msg_set		  ; Set message display duration

; Fall into the Main Program Loop

;==============================================================================
;
; Main Program loop
;
; The configuration of the PIC and, if selected, the IF setting mode has been completed
; in the previous routine. The main program is a loop which sequentially prepares the
; variable and calls the measurement and display routines.  
;
; The counter input uses the asynchronous register on Port A4, the TMR0 register and
; the freq_2 register to do a binary count of the input frequency. Every time the TMR0
; register overflows freq_2 is incremented. The pre-scaler performs a divide by 256 action
; on the measured frequency. 

; The resulting 32 bit binary word is transferred to freq0..freq2 where it is used by the 
; Binary to BCD routine for direct display or the IF offset and/or multiply routines before 
; which modify the 32 bit word before being used by the Binary to BCD routine.
;
; The option to enable direct or IF offset display mode is contained within the loop allowing
; this feature to be changed "on the fly". The IF Add/Subtract function resides in the
; IF Offset routine.
;==============================================================================

; Start of main loop 

main
	movlw	0x80		; Load message display timer (128 cycles ~ 15 secs)
	movwf	msg_timer	
	call	Null_msg	; Blank line 2 with null message
main1				; Clear all counters		
	movf	msgflag,w
	movwf	msgflag_old	; Record msg flag status for comparison later in routine
	clrf 	freq_0
	clrf 	freq_1
	clrf 	freq_2
	clrf 	freq_3
	clrf 	TMR0		; Clear TMR0 and Prescaler
	call 	gate		; Start measurement
	call	multiply	; Apply frequency multiplier factor (x_freq)
	btfsc	PortA,IF_En	; Use IF Offset?
	goto	Offset
	call	Direct_msg
	goto	No_offset
Offset
	call	IF_Offset
No_offset
	call	bin2BCD		; Convert Binary count into BCD
	call	show_freq	; Display frequency
	decfsz	msg_timer,f	; Decrement message display timer every count cycle
	goto	main1		; Loop back to make next measurment
	btfss	msg_state,0	; Test if timed message blanking needed (1=No, 0=Yes)
	call	Null_msg	; Blank line 2 with null message
	incf	msg_timer,f	; Step counter up one ready for next decrement to hold it at zero
	movf	msgflag,w	; Load msgflag byte into w
	xorwf	msgflag_old,w	; Has it changed?
	btfss	STATUS,B_Z
	goto	main1		; No
	goto	main		; Yes, switch on message display again
;
;
;==============================================================================
;
; Programming Mode
;
; If this mode is selected on Power On the IF Offset Frequency is programmed via 
; the switches on Port A. On exiting programming mode the new IF Offset is saved
; to the EEPROM.
;
; On completion of IF Offset programming the option to use frequency multiplication
; is offered. The multiplication factor is programmed using the Up/Down buttons
; and saved in EEPROM on exit.
;
; IF_n is used for decade step value
; freq_n is used to hold new IF frequency
;
;==============================================================================

IF_Set
	movf	IF_0,w		; Save IF value to freq
	movwf	freq_0		; to simplify re-use of variables
	movf	IF_1,w		; in IF Set calculations
	movwf	freq_1
	movf	IF_2,w
	movwf	freq_2
	movf	IF_3,w
	movwf	freq_3	
;
; Check to make sure that the buttons have been released. The exit and save command is
; again both buttons pressed, this check holds the program here on entry to prevent a  
; pass and exit before any programming.
;

	call 	show		; Display IF value, this prompts user that we've now entered IF_Set routine
	movfw	PortA		; Get Port A contents
	andlw	b'11000'	; Mask Push button states
	bcf	STATUS,B_Z	; Clear Zero flag
	sublw	b'00001000'	; If buttons released we will get a zero (RA4 Normal = 0, RA3 Normal = 1)
	btfss	STATUS,B_Z	;
	goto	IF_Set		; Not fully released jump back	

Range
	clrf	IF_Range
	movfw 	PortA		; Get setting of 3 bit Binary Range Switch
	andlw	b'00111'	; Mask to extract switch setting
	movwf	IF_Range	; Place result in IF_Range
	rlf	IF_Range,f	; Step data is in blocks of 4 bytes, the address needs
	rlf	IF_Range,f	; to be multiplied by 4 to point to block, hence shift left twice
	movf	IF_Range,w	; Put result in w for addressing step size table
	call	Step_Table	; Get IF Step Size as four bytes
	movwf	IF_0		; Place LSB constant in Register
	incf	IF_Range,f	; Point to next byte
	movf	IF_Range,w	; 
	call	Step_Table	; This makes re-use of IF terms
	movwf	IF_1		; to simplify use with IF Add/Sub Routine
	incf	IF_Range,f	; 
	movf	IF_Range,w	; 
	call	Step_Table	; 
	movwf	IF_2		; 
	incf	IF_Range,f	; 
	movf	IF_Range,w	; 
	call	Step_Table	; 	
	movwf	IF_3		; Place MSB constant in Register
scan
	movlw	0x08		; Load Delay Counter
	movwf	count
scan1		
; Check for exit condition
	movfw	PortA		; Get Port A contents
	andlw	b'11000'	; Mask Push button states
	bcf	STATUS,B_Z	; Clear Zero flag
	sublw	b'00010000'	; If both buttons pushed we will get a zero (RA4 Normal = 0, RA3 Normal = 1)
	btfsc	STATUS,B_Z	;
	return
	btfsc	PortA, PB_Up	; Scan Step Up Switch				(Normal 0, Operated 1)
	call	stepup
	btfss	PortA, PB_Down	; Scan Step Down Switch				(Normal 1, Operated 0)
	call	stepdown
	call	show		; Display
	goto	Range		; Back to decade switch scan
;
; Increment the IF decade
;

stepup
	call	wait_64ms	; Debounce
	btfss	PortA, PB_Up	; Check						(Normal 0, Operated 1)
	goto	endup		; False, jump back to beginning
	btfss	PortA, PB_Down	; Check to see if we need to exit and save	(Normal 1, Operated 0)
	return			; Leave
	decfsz	count,f
	goto	stepup		; Loop back
	call	add_IF		; Add Step
endup
	return

;
; Decrement the IF decade
;

stepdown
	call	wait_64ms	; Debounce
	btfsc	PortA, PB_Down	; Check						(Normal 1, Operated 0)
	goto	enddown		; False, jump back to beginning
	btfsc	PortA, PB_Up	; Check to see if we need to exit and save	(Normal 0, Operated 1)
	return			; Leave
	decfsz	count,f
	goto	stepdown	; Loop back
	call	sub_IF		; Subtract Step
enddown
	return

;=============================================================================
; Routines associated with the Programming Mode
;=============================================================================

;
; Display the IF frequency
;

show
	call	bin2BCD		; Convert Binary count into BCD
	call	show_freq	; Display frequency
	call	Prog_msg
	return

;=============================================================================
; Frequency Multiplier set routine.
;=============================================================================

mult_set
			
; Check to make sure that the buttons have been released. The exit and save command is
; again both buttons pressed, this check holds the program here on entry to prevent a  
; pass and exit before any programming.

	call button_test	; Check both buttons normal

scan_x		
	movlw	0x04		; Load Delay Counter
	movwf	count
	btfsc	PortA, PB_Up	; Scan Step Up Switch				(Normal 0, Operated 1)
	call	stepup_x
	btfss	PortA, PB_Down	; Scan Step Down Switch				(Normal 1, Operated 0)
	call	stepdown_x
	call	bin2BCDx	; Convert x_freq binary value to BCD
	call	show_x_factor	; Display factor
	call	Prog_x_msg	; Display mode message
	goto	scan_x
	
; Incremement freq mult factor

stepup_x
	call	wait_64ms	; Debounce
	btfss	PortA, PB_Up	; Check						(Normal 0, Operated 1)
	goto	endup_x		; False, jump back to beginning
	btfss	PortA, PB_Down	; Check to see if we need to exit and save	(Normal 1, Operated 0)
	goto 	msg_set		; Leave
	decfsz	count,f
	goto	stepup_x	; Loop back
	incf	x_freq,f	; Increment and check for greater than 99 (63h)
	movlw	0x63		; Load w with 64h
	subwf	x_freq,w	; Subtract from x_freq
	btfss	STATUS,B_C	; x_freq>63h?
	goto	endup_x		; No
	movlw	0x63		; Yes, load w with 63h
	movwf	x_freq		; Set x_freq to 99
endup_x
	return

; Decrement freq mult factor

stepdown_x
	call	wait_64ms	; Debounce
	btfsc	PortA, PB_Down	; Check						(Normal 1, Operated 0)
	goto	enddown_x	; False, jump back to beginning
	btfsc	PortA, PB_Up	; Check to see if we need to exit and save	(Normal 0, Operated 1)
	goto	msg_set		; Leave
	decfsz	count,f
	goto	stepdown_x	; Loop back
	decfsz	x_freq,f	; Decrement and check for zero
	goto	enddown_x	; Not zero
	clrf	x_freq		; Set x_freq to zero
enddown_x
	return	

; Convert binary x_freq byte to single BCD byte for display

bin2BCDx
	clrf	x_BCD		; Reset x_freq BCD value
	movf	x_freq,w	
	movwf	temp		; Get x_freq value
ms_nibble
	bcf	STATUS,B_C	; Clear Carry bit
	movlw	0x0A		; Load w with 10
	subwf	temp,w		; Subtract from x_freq 
	btfss	STATUS,B_C	; Has it gone negative?
	goto	ls_nibble	; Yes, content is <10
	movwf	temp		; No, put new value into temp 
	movlw	0x10		; Load 00010000 into w	
	addwf	x_BCD,f		; Increment ms nibble
	goto	ms_nibble	; Jump back for another pass
ls_nibble
	movf	temp,w		; Put <10 value into w
	addwf	x_BCD,f		; Add to BCD value
	Return			; Finished

show_x_factor
        movlw   0x82              ; Point the LCD to first LCD digit location
        call    cmnd2LCD          ; Send starting digit location to LCD
        swapf   x_BCD,w           ; Swap MS BCD digit into lower nibble of W
        andlw   0x0F              ; Mask for lower nibble only       (0000XXXX)
        addlw   0x30              ; Add offset for ASCII char set    (0030XXXX)
        call    data2LCD          ; Send data byte in W to LCD
        movf    x_BCD,w           ; Put LS BCD digit into lower nibble of W
        andlw   0x0F              ; Mask for lower nibble only       (0000YYYY)
        addlw   0x30              ; Add offset for ASCII char set    (0030YYYY)
        call    data2LCD          ; Send byte in W to LCD
	return

;
; Set permanent or temporary message display on line 2
;
;=============================================================================
; Set Line 2 Display duration
;=============================================================================

msg_set

; Check to make sure that the buttons have been released. The exit and save command is
; again both buttons pressed, this check holds the program here on entry to prevent a  
; pass and exit before any programming. The two messages are the same length and so
; overwrite each other hence no Null message is needed to blank the line before display.

	call	button_test	; Check both buttons are normal

scan_msg		
	clrf	msgflag1	; Clear overwriting flag
	movlw	0x0A		; Load counter with 10
	movwf	count
	btfsc	PortA, PB_Up	; Scan Step Up Switch				(Normal 0, Operated 1)
	call	set_msg_on	; Set Display permanent
	btfss	PortA, PB_Down	; Scan Step Down Switch				(Normal 1, Operated 0)
	call	set_msg_off	; Set display timed
	goto	scan_msg	; Jump back

set_msg_on
	call	wait_64ms	; Debounce
	btfss	PortA, PB_Up	; Check						(Normal 0, Operated 1)
	return			; False, return
	bsf	msg_state,0	; Set message status to on (1)
	call	Msg_on_msg	; Display status
	decfsz	count,f
	goto	set_msg_on	; Jump back
	goto 	Save		; Leave
	
set_msg_off
	call	wait_64ms	; Debounce
	btfsc	PortA, PB_Down	; Check						(Normal 1, Operated 0)
	return			; False, return
	bcf	msg_state,0	; Set message status to off (0)
	call	Msg_off_msg	; Display status
	decfsz	count,f
	goto	set_msg_off	; Jump back
	goto	Save		; Leave
;=============================================================================
; Common code
;=============================================================================

button_test
	movfw	PortA		; Get Port A contents
	andlw	b'11000'	; Mask Push button states
	bcf	STATUS,B_Z	; Clear Zero flag
	sublw	b'00001000'	; If buttons released we will get a zero (RA4 Normal = 0, RA3 Normal = 1)
	btfss	STATUS,B_Z	;
	goto	button_test	; Not fully released jump back
	return


; Once the new IF value, the frequency multiplier and display config have been 
; set the values are written to EEPROM and the program drops into a parking loop. 
; Normal operation is restored and the new values are activated after 
; powering on and off. 

Save
	call 	Write		; Save IF and x_freq values
	call	Stored_msg	; Display "Saved" message
park	nop
	goto 	park		; Parking loop - Power off/on reset

;=============================================================================
;
;These routines save the new IF frequency, held in freq_n, to EEPROM
;
;=============================================================================

Write				  ; Save new IF and freq factor values to EEPROM

        clrf    EEadr             ; Reset EEPROM Address 
        movf    freq_0,w          ; Record the first
        movwf   EEdata            ;   IF
        call    write_EEPROM      ;   byte
        movf    freq_1,w          ; Record the second
        movwf   EEdata            ;   IF
        call    write_EEPROM      ;   byte
        movf    freq_2,w          ; Record the third
        movwf   EEdata            ;   IF
        call    write_EEPROM      ;   byte
        movf    freq_3,w          ; Record the fourth
        movwf   EEdata            ;   IF
        call    write_EEPROM      ;   byte
        movf    x_freq,w          ; Record the fifth (x_freq value)
        movwf   EEdata            ; x_freq  
        call    write_EEPROM      ; byte
	movf	msg_state,w	  ; Record the sixth (message perm/timed)
        movwf   EEdata		  ; Message Status
	call	write_EEPROM  
        return                    ; Return to the caller
;
; *****************************************************************************
; *                                                                           *
; * Purpose:  Write the byte of data at EEdata to the EEPROM at address       *
; *           EEadr.                                                          *
; *                                                                           *
; *   Input:  The values at EEdata and EEadr.                                 *
; *                                                                           *
; *  Output:  The EEPROM value is updated.                                    *
; *                                                                           *
; *****************************************************************************
;
write_EEPROM
	Bank1		          ; Switch to bank 1
        bsf     EEdata,WREN       ; Set the EEPROM write enable bit
        movlw   0x55              ; Write 0x55 and 0xAA to EEPROM
        movwf   EEadr             ; control register, as required
        movlw   0xAA              ; for the write
        movwf   EEadr             ; 
        bsf     EEdata,WR         ; Set WR to initiate write
bit_check
        btfsc   EEdata,WR         ; Has the write completed?
        goto    bit_check         ; No, keep checking
        bcf     EEdata,WREN       ; Clear the EEPROM write enable bit
	Bank0			  ; Switch to bank 0 
        incf    EEadr,f           ; Increment the EE write address
        return                    ; Return to the caller
;==============================================================================
;
; Read IF Offset and Freq Multiplier data from EEPROM
;
;
;
;
;
;==============================================================================
;

read_EEifs
        clrf    EEadr             ; Reset the EEPROM read address
        call    read_EEPROM       ; Read EEPROM
        movf    EEdata,w          ; Get the first byte
        movwf   IF_0	          ; Save IF frequency
        call    read_EEPROM       ; Get next byte
        movf    EEdata,w          ; 
        movwf   IF_1              ; Save it
        call    read_EEPROM       ; Get the third byte
        movf    EEdata,w          ; 
        movwf   IF_2              ; Save it
        call    read_EEPROM       ; Get the fourth byte
        movf    EEdata,w          ;
        movwf   IF_3              ; Save it
	call    read_EEPROM       ; Get the Frequency Multiplier byte
        movf    EEdata,w          ;
        movwf   x_freq            ; Save it
	call    read_EEPROM       ; Get the Message timed/perm status
        movf    EEdata,w          ;
        movwf   msg_state         ; Save it
	return
;
; *****************************************************************************
; *                                                                           *
; * Purpose:  Read a byte of EEPROM data at address EEadr into EEdata.        *
; *                                                                           *
; *   Input:  The address EEadr.                                              *
; *                                                                           *
; *  Output:  The value in EEdata.                                            *
; *                                                                           *
; *****************************************************************************

read_EEPROM
        Bank1		          ; Switch to bank 1
        bsf     EEdata,RD         ; Request the read
        Bank0		          ; Switch to bank 0
        incf    EEadr,f           ; Increment the read address
        return                    ; Return to the caller

; *****************************************************************************
; *                                                                           *
; * Purpose:  Power on initialization of Liquid Crystal Display.  The LCD     *
; *           controller chip must be equivalent to an Hitachi 44780.  The    *
; *           LCD is assumed to be a 16 X 2 display.                          *
; *                                                                           *
; *   Input:  None                                                            *
; *                                                                           *
; *  Output:  None                                                            *
; *                                                                           *
; *****************************************************************************
;
init_LCD
        call    wait_64ms         ; Wait for LCD to power up
        movlw   0x30              ; LCD init instruction (First)
        bsf     PortB,LCD_e       ; Set the LCD E line high,
	movwf   PortB             ; Send to LCD via RB7..RB0
        call    wait_64ms         ;   wait a "long" time,
        bcf     PortB,LCD_e       ;   and then Clear E 
        movlw   0x30              ; LCD init instruction (Second)
        bsf     PortB,LCD_e       ; Set E high,
        movwf   PortB             ; Send to LCD via RB7..RB0
        call    wait_32ms         ;   wait a while,
        bcf     PortB,LCD_e       ;   and then Clear E 
        movlw   0x30              ; LCD init instruction (Third)
        bsf     PortB,LCD_e       ; Set E high,
	movwf   PortB             ; Send to LCD via RB7..RB0
        call    wait_32ms         ;   wait a while,
        bcf     PortB,LCD_e       ;   and then Clear E
        movlw   0x20              ; 4-bit mode instruction, 
        bsf     PortB,LCD_e       ; Set E high,
        movwf   PortB             ; Send to LCD via RB7..RB0
        call    wait_16ms         ;   wait a while,
        bcf     PortB,LCD_e       ;   and then Clear E
        movlw   0x28              ; 1/16 duty cycle, 5x8 matrix
        call    cmnd2LCD          ; Send command in w to LCD
        movlw   0x08              ; Display off, cursor and blink off 
        call    cmnd2LCD          ; Send command to LCD
        movlw   0x01              ; Clear and reset cursor
        call    cmnd2LCD          ; Send command in w to LCD
        movlw   0x06              ; Set cursor to move right, no shift
        call    cmnd2LCD          ; Send command in w to LCD
        movlw   0x0C              ; Display on, cursor and blink off
        call    cmnd2LCD          ; Send command in w to LCD
        return                    ; 
;==========================================================================================
; GATE ROUTINE
;
; The gate routine loads the delay counters, opens the prescalar for a 100mSec period
; and checks for TMRO overflow by testing the INTCON bit. On overlflow the LSB
; in freq_2 is incremented. On completion TMR0 is shifted into freq_1 and the prescaler 
; contents are counted into freq_0.
;
; The delay is a made by a nested couter routine using
; delay1 and delay2 values. At the end of the count the binary frequency is held in the 
; prescalar, TMRO and Hibyte.
;
; 
; Delay is 250*10 machine cycles*40*5  = 100,000uSec for 4MHz clock
;
;==========================================================================================

gate			; Load up delays

	movlw	0x28	; Load outer loop with 40
	movwf	delay2 	; Gate outer loop delay   
	movlw	0xF9 	; Load with 248
	movwf	delay1 	; Gate inner delay

; Prepare Gate

        Bank1			; Switch to Bank 1
        movlw      b'00010011' 	; RA0,1..RA4 input, RA2,3 output
        movwf      TRISA
	Bank0			; Switch to Bank 0
	bcf	INTCON,2
	bsf	PortA,Clock
	bsf	PortA,PB_Down	; Gate Open

Loop1			 	; Machine Cycles (Total by branch)
        btfss   INTCON,2 	; 1   	Test for TMR0 overflow
        goto    Nocarry  	; 3
        incf    freq_2,f 	; 3	Count in TMR0 exceeds FF, increment freq_2
	bcf	INTCON,2	; 4	Reset TMR0 interrupt flag
	goto	Delay		; 6
Nocarry     
        nop              	; 4	Dummy cycles to balance delay in both branches
        nop		 	; 5
	nop			; 6	
Delay       
	nop			; 7
	decfsz  delay1,f   	; 8
        goto    Loop1     	; 10	248*10uSec
	movlw	0xF9 		; 10	Reload Gate inner delay with 248
	movwf	delay1 		; 11
        btfss   INTCON,2 	; 12   	Test for TMR0 overflow
        goto    Nocarry1  	; 14
        incf    freq_2,f 	; 14	Count in TMR0 exceeds FF, increment freq_2
	bcf	INTCON,2	; 15	Reset TMR0 interrupt flag
	goto	Delay2		; 17
Nocarry1  
        nop              	; 15	Dummy cycles to balance delay in both branches
        nop		 	; 16
	nop			; 17
Delay2
	decfsz	delay2,f 	; 18	Outer Loop check, when zero jump out to stop
	goto	Loop1	 	; 20	Else continue gate
stop
	bcf	PortA,PB_Down	; 20  Close Gate
	bcf	PortA,Clock
        btfss   INTCON,2 	;     Final test for TMR0 overflow
        goto    Nocarry2  	; 
        incf    freq_2,f 	;     Count in prescaler and TMRO exceeds FF, increment freq_2

; Move contents of TMR0 into freq_1
Nocarry2
	movf       TMR0,w
	movwf      freq_1 

; Move contents of prescaler into Freq_0. This cannot be done directly and any form of write
; action on TMR0 to evaluate prescaler contents will reset the prescaler.
; The contents are shifted by counting increments until a change is seen in the TMR0 contents
; Each increment is counted by a down counter, when TMR0 changes the down conter = prescaler content.
; Error in prescaler evaluation, sometimes wrong qrg (up 2,56KHz);bug solved by DL5MGD
	movf	   TMR0,w
	movwf	   temp		; Save value for comparison
	movlw	   0xff
	movwf	   count 	; Load count with ffh
	bcf	   STATUS,B_Z	; Clear Zero flag
PCount
	decfsz	   count,f	; Count down
        goto	   nxtPSup
	goto	   set0	
nxtPSup	bsf        PortA,Clock  ; Clock Hi
        bcf        PortA,Clock  ; Clock Lo
	movf 	   temp,w
	subwf	   TMR0,w
        btfsc      STATUS,B_Z	; Did we change from zero?
        goto       PCount	; No, go round again
set0    movf	   count,w	; Yes, count = prescaler content
	movwf	   freq_0
			
        return

;==========================================================================================
; Frequency Multiplier Routine
;
; This routine multiplies the binary count by the integer x_freq value. For direct 
; frequency measurement x_freq will be set to 1, for applications involving an 
; oscillator multiplier chain x_freq can be set to the multiplication factor allowing
; display of final frequency with or without an IF offset.
;
; Typical applications for this feature would be VXO sources for VHF transceivers.
;
;
;	Input	binary count in freq_n, Freq Mult factor x_freq
;
;	Output	multiplied binary count in freq_n
;
;
;==========================================================================================
multiply
	movf 	x_freq,w	 ; Get x_freq factor
	movwf	count		 ; Place in counter
	clrf	xfreq_0		 ; Reset temporary storage
	clrf	xfreq_1
	clrf	xfreq_2
	clrf	xfreq_3
add
        movf    freq_0,w         ; Get low byte of freq
        addwf   xfreq_0,f      	 ; Add it to the low byte of freq
        btfss   STATUS,B_C       ; Any carry?
        goto    add1             ; No, add next byte
        incfsz  xfreq_1,f        ; Ripple carry up to the next byte
        goto    add1             ; No new carry, add next byte
        incfsz  xfreq_2,f        ; Ripple carry up to the next byte
        goto    add1             ; No new carry, add next byte
        incf    xfreq_3,f        ; Ripple carry up to the highest byte
add1
        movf    freq_1,w         ; Get the next byte
        addwf   xfreq_1,f        ; Add it to itself
        btfss   STATUS,B_C       ; Any carry?
        goto    add2             ; No, add next byte
        incfsz  xfreq_2,f        ; Ripple carry up to the next byte
        goto    add2             ; No new carry, add next byte
        incf    xfreq_3,f        ; Ripple carry up to the highest byte
add2
        movf    freq_2,w         ; Get the next to most significant byte
        addwf   xfreq_2,f        ; Add it to itself
        btfss   STATUS,B_C       ; Any carry?
        goto    add3             ; No, add last byte
        incf    xfreq_3,f        ; Ripple carry up to the highest byte
add3
        movf    freq_3,w         ; Get the most significant byte
        addwf   xfreq_3,f        ; Add it to itself
	decfsz	count,f
 	goto	add		 ; Jump back for next pass
	movf	xfreq_0,w	 ; Transfer result back into freq_n
	movwf	freq_0
	movf	xfreq_1,w	 ; 
	movwf	freq_1
	movf	xfreq_2,w	 ; 
	movwf	freq_2
	movf	xfreq_3,w	 ; 
	movwf	freq_3
	return			 ; Finished

;==========================================================================================
; IF Offset Routine
; 
; This uses modified versions of the 32 bit add and subtract routines to shift
; the DDS frequency data by the value held if the IF_Add and IF_Sub constants.
; If RA4 is high the IF frequency in Hz is added, if RA4 is low the IF 
; frequency is subtracted. Negative results in the subtraction process are 
; made positive by a two's compliment subtroutine. This produces a positive
; value which is the difference between the displayed frequency and the IF
; frequency.
;
;==========================================================================================
;
;
IF_Offset			

	 btfss 	PortA,IF_AS	; Test state of IF offset switch
	 goto 	sub_IF		; Low so subtract
	 goto	add_IF		; High so add 
IF_exit1 return			; Return to Main


;==========================================================================================
;  
;  add_IF
;                                                                          
;  Purpose:  This routine adds the 32 bit value of fstep to the 32 bit       
;            value in freq.                                
;                                                                            
;    Input:  The 32 bit values in IF and freq                             
;                                                                            
;   Output:  The sum of IF and freq is stored in freq.                
;                                                                            
;==========================================================================================

add_IF
	call	Offset_sum_msg	 ; Mode Message
        movf    IF_0,w           ; Get low byte of the increment
        addwf   freq_0,f       ; Add it to the low byte of freq
        btfss   STATUS,B_C         ; Any carry?
        goto    IFadd1             ; No, add next byte
        incfsz  freq_1,f       ; Ripple carry up to the next byte
        goto    IFadd1             ; No new carry, add next byte
        incfsz  freq_2,f       ; Ripple carry up to the next byte
        goto    IFadd1             ; No new carry, add next byte
        incf    freq_3,f       ; Ripple carry up to the highest byte
IFadd1
        movf    IF_1,w           ; Get the next increment byte
        addwf   freq_1,f       ; Add it to the next higher byte
        btfss   STATUS,B_C         ; Any carry?
        goto    IFadd2             ; No, add next byte
        incfsz  freq_2,f       ; Ripple carry up to the next byte
        goto    IFadd2             ; No new carry, add next byte
        incf    freq_3,f       ; Ripple carry up to the highest byte
IFadd2
        movf    IF_2,w           ; Get the next to most significant increment
        addwf   freq_2,f       ; Add it to the freq byte
        btfss   STATUS,B_C         ; Any carry?
        goto    IFadd3             ; No, add last byte
        incf    freq_3,f       ; Ripple carry up to the highest byte
IFadd3
        movf    IF_3,w           ; Get the most significant increment byte
        addwf   freq_3,f       ; Add it to the most significant freq
        goto 	IF_exit1           ; Finished
;==========================================================================================
; 
;  sub_IF
;                                                                          
;  Purpose:  Subtract the IF from freq, checking that it does               
;            not go below zero.                                             
;                                                                           
;    Input:  The values in IF and freq.                                     
;                                                                           
;   Output:  The updated value in freq.                                     
;                                                                           
;
;==========================================================================================
sub_IF
	call	Offset_diff_msg	 ; Mode Message
        comf    freq_0,f         ; Subtraction of fstep from
        comf    freq_1,f         ; freq is done by adding the
        comf    freq_2,f         ; twos compliment of freq
        comf    freq_3,f         ; to IF
        incfsz  freq_0,f         ; Increment last byte
        goto    IF_comp_done         ; Non-zero, continue
        incfsz  freq_1,f         ; Increment next byte
        goto    IF_comp_done         ; Non-zero, continue
        incfsz  freq_2,f         ; Increment next byte
        goto    IF_comp_done         ; Non-zero, continue
        incf    freq_3,f         ; Increment the high byte
IF_comp_done
;
;	Now add the compliment of freq and IF
;
	movf    IF_0,w           ; Get low byte of the increment
        addwf   freq_0,f       ; Add it to the low byte of freq
        btfss   STATUS,B_C         ; Any carry?
        goto    IFSadd1            ; No, add next byte
        incfsz  freq_1,f       ; Ripple carry up to the next byte
        goto    IFSadd1             ; No new carry, add next byte
        incfsz  freq_2,f       ; Ripple carry up to the next byte
        goto    IFSadd1             ; No new carry, add next byte
        incf    freq_3,f       ; Ripple carry up to the highest byte
IFSadd1
        movf    IF_1,w           ; Get the next increment byte
        addwf   freq_1,f       ; Add it to the next higher byte
        btfss   STATUS,B_C         ; Any carry?
        goto    IFSadd2            ; No, add next byte
        incfsz  freq_2,f       ; Ripple carry up to the next byte
        goto    IFSadd2             ; No new carry, add next byte
        incf    freq_3,f       ; Ripple carry up to the highest byte
IFSadd2
        movf    IF_2,w           ; Get the next to most significant increment
        addwf   freq_2,f       ; Add it to the freq byte
        btfss   STATUS,B_C         ; Any carry?
        goto    IFSadd3            ; No, add last byte
        incf    freq_3,f       ; Ripple carry up to the highest byte
IFSadd3
        movf    IF_3,w           ; Get the most significant increment byte
        addwf   freq_3,f       ; Add it to the most significant freq
;
;       If the frequency has gone negative, two's compliment it.
;
        btfss   freq_3,7      ; Is high order frequency byte "negative"?   
        goto    IF_exit1          ; No, keep going
IF_comp
        comf    freq_0,f          ; Yes, two's compliment result
        comf    freq_1,f          ; to make positive difference
        comf    freq_2,f          ; 
        comf    freq_3,f          ; 
	incfsz  freq_0,f          ; Increment last byte
        goto    IF_exit2              ; Non-zero, continue
        incfsz  freq_1,f          ; Increment next byte
        goto    IF_exit2              ; Non-zero, continue
        incfsz  freq_2,f          ; Increment next byte
        goto    IF_exit2              ; Non-zero, continue
        incf    freq_3,f          ; Increment the high byte
IF_exit2
        goto IF_exit1                 ; Subtract finished
;
; *****************************************************************************
; *                                                                           *
; * Purpose:  This subroutine converts a 32 bit binary number to a 10 digit   *
; *           BCD number.  The input value taken from freq(0 to 3) is         *
; *           preserved.  The output is in BCD(0 to 4), each byte holds =>    *
; *           (hi_digit,lo_digit), most significant digits are in BCD_4.      *
; *           This routine is a modified version of one described in          *
; *           MicroChip application note AN526.                               *
; *                                                                           *
; *   Input:  The value in freq_0 ... freq_3                                  *
; *                                                                           *
; *  Output:  The BCD number in BCD_0 ... BCD_4                               *
; *                                                                           *
; *****************************************************************************
;
bin2BCD
        movlw   0x20              ; Set loop counter
        movwf   BCD_count         ;   to 32
        clrf    BCD_0             ; Clear output
        clrf    BCD_1             ;   "     "
        clrf    BCD_2             ;   "     "
        clrf    BCD_3             ;   "     "
        clrf    BCD_4             ;   "     "

bin_loop
        bcf     STATUS,B_C        ; Clear carry bit in STATUS
;
; Rotate bits in freq bytes.  Move from LS byte (freq_0) to next byte (freq_1).
; Likewise, move from freq_1 to freq_2 and from freq_2 to freq_3.
;
        rlf     freq_0,f          ; Rotate left, 0 -> LS bit, MS bit -> Carry
        rlf     freq_1,f          ; Rotate left, Carry->LS bit, MS bit->Carry
        rlf     freq_2,f          ; Rotate left, Carry->LS bit, MS bit->Carry
        rlf     freq_3,f          ; Rotate left, Carry->LS bit, MS bit->Carry
        btfsc   STATUS,B_C        ; Is Carry clear? If so, skip next instruction
        bsf     freq_0,0          ; Carry is set so wrap and set bit 0 in freq_0
;
; Build BCD bytes. Move into LS bit of BCD bytes (LS of BCD_0) from MS bit of
; freq_3 via the Carry bit.  
;
        rlf     BCD_0,f           ; Rotate left, Carry->LS bit, MS bit->Carry
        rlf     BCD_1,f           ; Rotate left, Carry->LS bit, MS bit->Carry
        rlf     BCD_2,f           ; Rotate left, Carry->LS bit, MS bit->Carry
        rlf     BCD_3,f           ; Rotate left, Carry->LS bit, MS bit->Carry
        rlf     BCD_4,f           ; Rotate left, Carry->LS bit, MS bit->Carry
        decf    BCD_count,f       ; Decrement loop count
        btfss   STATUS,B_Z        ; Is loop count now zero?
        goto    adjust            ; No, go to adjust
        return                    ; Yes, EXIT 
; ============================================================================
adjust  ; Internal subroutine, called by bin2BCD main loop only
; 
; As BCD bytes are being built, make sure the nibbles do not grow larger than 9. 
; If a nibble gets larger than 9, increment to next higher nibble.  
; (If the LS nibble of a byte overflows, increment the MS nibble of that byte.)
; (If the MS nibble of a byte overflows, increment the LS nibble of next byte.)
;
        movlw   BCD_0             ; Get pointer to BCD_0
        movwf   FSR               ; Put pointer in FSR for indirect addressing
        call    adj_BCD           ; 
        incf    FSR,f             ; Move indirect addressing pointer to BCD_1
        call    adj_BCD           ; 
        incf    FSR,f             ; Move indirect addressing pointer to BCD_2
        call    adj_BCD           ; 
        incf    FSR,f             ; Move indirect addressing pointer to BCD_3
        call    adj_BCD           ; 
        incf    FSR,f             ; Move indirect addressing pointer to BCD_4
        call    adj_BCD           ; 
        goto    bin_loop          ; Back to main loop of bin2BCD
; ============================================================================
adj_BCD  ; Internal subroutine, called by adjust only
        movlw   3                 ; Add 3
        addwf   INDF,w            ;   to LS digit
        movwf   BCD_temp          ; Save in temp
        btfsc   BCD_temp,3        ; Is LS digit + 3 > 7  (Bit 3 set)
        movwf   INDF              ; Yes, save incremented value as LS digit
        movlw   0x30              ; Add 3
        addwf   INDF,w            ;   to MS digit
        movwf   BCD_temp          ; Save as temp
        btfsc   BCD_temp,7        ; Is MS digit + 3 > 7  (Bit 7 set)
        movwf   INDF              ; Yes, save incremented value as MS digit
        return                    ; Return to adjust subroutine
;
; *****************************************************************************
; *                                                                           *
; * Purpose:  Display the frequency setting on the LCD.                       *
; *                                                                           *
; *   Input:  The values in BCD_4 ... BCD_0                                   *
; *                                                                           *
; *  Output:  The number displayed on the LCD                                 *
; *                                                                           *
; *****************************************************************************
;
;
; Note that because of 100mSec gate period the decades are one tenth of their
; actual frequency (1sec measurement). Allocation of digits to display positions
; corrects this - e.g. 100kHz = 1MHz 
;
;
show_freq
        movlw   0x82              ; Point the LCD to first LCD digit location
        call    cmnd2LCD          ; Send starting digit location to LCD
	clrf	zeroflag	  ; Clear register used to record leading zero states
;
; Running 4-bit mode, so need to send Most Significant Nibble first.
;
; Extract and send "XXXX" from byte containing "XXXXYYYY"
;  - Swap halves to get YYYYXXXX
;  - Mask with 0x0F to get 0000XXXX
;  - Add ASCII bias (0030XXXX)
;


D100MHz
        swapf   BCD_3,w           ; Swap 100MHz BCD digit into lower nibble of W
        andlw   0x0F              ; Mask for lower nibble only       (0000XXXX)
	btfsc	STATUS,B_Z	  ; Is the digit zero?
	goto	D100Blank	  ; Yes
        addlw   0x30              ; No, Add offset for ASCII char set    (0030XXXX)
        call    data2LCD          ; Send byte in W to LCD
	goto	D10MHz		  ; Next digit
D100Blank
	bsf	zeroflag,0	  ; Record that 100MHz was a zero
	movlw	' '		  ; Load code for space
	call	data2LCD
;
; Extract and send "YYYY" from byte containing "XXXXYYYY"
;   - Mask with 0x0F to get 0000YYYY
;   - Add offset for ASCII character set in LCD  (0030YYYY)
;
D10MHz
        movf    BCD_3,w           ; Put 10MHz BCD digit into lower nibble of W
        andlw   0x0F              ; Mask for lower nibble only       (0000XXXX)
	btfsc	STATUS,B_Z	  ; Is the digit zero?
	goto	D10Blank	  ; Yes
D10NoBlank
        addlw   0x30              ; Add offset for ASCII char set    (0030XXXX)
        call    data2LCD          ; Send byte in W to LCD
	goto	D1MHz		  ; Next digit
D10Blank
	btfss	zeroflag,0	  ; Was previous digit also zero?
	goto	D10NoBlank	  ; No
	bsf	zeroflag,1	  ; Record that 10MHz was a zero
	movlw	' '		  ; Load code for space
	call	data2LCD
;
D1MHz
        swapf   BCD_2,w           ; Swap 1MHz BCD digit into lower nibble of W
        andlw   0x0F              ; Mask for lower nibble only       (0000XXXX)
	btfsc	STATUS,B_Z	  ; Is the digit zero?
	goto	D1Blank	  	  ; Yes
D1NoBlank
        addlw   0x30              ; Add offset for ASCII char set    (0030XXXX)
        call    data2LCD          ; Send byte in W to LCD
	goto	Decimal_Point
D1Blank
	btfss	zeroflag,1	  ; Was previous digit also zero?
	goto	D1NoBlank	  ; No
	bsf	zeroflag,2	  ; Record that 1MHz was a zero
	movlw	' '		  ; Load code for space
	call	data2LCD
	movlw	' '
	call	data2LCD	  ; Freq has no MHz Component, skip decimal point and insert space	
	goto	D100kHz		  ; 
;
Decimal_Point
        movlw   '.'               ; Get a period
        call    data2LCD          ; Send byte in W to LCD
;
D100kHz
        movf    BCD_2,w           ; Put 100kHz BCD digit into lower nibble of W
        andlw   0x0F              ; Mask for lower nibble only       (0000YYYY)
        addlw   0x30              ; Add offset for ASCII char set    (0030YYYY)
        call    data2LCD          ; Send byte in W to LCD
;
; 10kHz
        swapf   BCD_1,w           ; Swap 10kHz BCD digit into lower nibble of W
        andlw   0x0F              ; Mask for lower nibble only       (0000XXXX)
        addlw   0x30              ; Add offset for ASCII char set    (0030XXXX)
        call    data2LCD          ; Send byte in W to LCD
;
; 1kHz
        movf    BCD_1,w           ; Put 1kHz BCD digit into lower nibble of W
        andlw   0x0F              ; Mask for lower nibble only       (0000YYYY)
        addlw   0x30              ; Add offset for ASCII char set    (0030YYYY)
        call    data2LCD          ; Send data byte in W to LCD
;
; Comma
        movlw   '.'               ; Set up W with ASCII comma 
        call    data2LCD          ; Send data byte in W to LCD
;
; 100Hz
        swapf	BCD_0,w           ; Swap 100Hz BCD digit into lower nibble of W
        andlw   0x0F              ; Mask for lower nibble only       (0000XXXX)
        addlw   0x30              ; Add offset for ASCII char set    (0030XXXX)
        call    data2LCD          ; Send data byte in W to LCD
;
; 10Hz
        movf   	BCD_0,w           ; Put 10Hz BCD digit into lower nibble of W
        andlw   0x0F              ; Mask for lower nibble only       (0000YYYY)
        addlw   0x30              ; Add offset for ASCII char set    (0030YYYY)
        call    data2LCD          ; Send byte in W to LCD

        movlw   ' '               ; Send a space
        call    data2LCD          ;   to LCD
;
	btfss	zeroflag,2	  ; Test to see if MHz or kHz
	goto	MHz

        movlw   'k'               ; Send a 'k'
        call    data2LCD          ;   to LCD
;
        movlw   'H'               ; Send an "H"
        call    data2LCD          ;   to LCD
;
        movlw   'z'               ; Send a 'z'
        call    data2LCD          ;   to LCD
;
        movlw   ' '               ; Send a space to blank "z" left on transition from MHz to kHz
        call    data2LCD          ;   to LCD
;
        return                    ; 

MHz
        movlw   'M'               ; Send a 'M'
        call    data2LCD          ;   to LCD
;
        movlw   'H'               ; Send an "H"
        call    data2LCD          ;   to LCD
;
        movlw   'z'               ; Send a 'z'
        call    data2LCD          ;   to LCD
;
        return                    ; 
;
; *****************************************************************************
; *                                                                           *
; * Purpose:  Check if LCD is done with the last operation.                   *
; *           This subroutine polls the LCD busy flag to determine if         *
; *           previous operations are completed.                              *
; *                                                                           *
; *   Input:  None                                                            *
; *                                                                           *
; *  Output:  PortB set as:  RB7..RB4 inputs                                  *
; *                          RB3..RB0 outputs                                 *
; *****************************************************************************
; 
busy_check
        clrf    PortB             ; Clear all outputs on PortB
        Bank1		          ; Switch to bank 1 for Tristate operation
        movlw   b'11110000'       ; Set RB7..RB4 as inputs, RB3..RB0 outputs
        movwf   TRISB             ;   via Tristate
        Bank0			  ; Switch back to bank 0
        bcf     PortB,LCD_rs      ; Set up LCD for Read Busy Flag (RS = 0) 
        bsf     PortB,LCD_rw      ; Set up LCD for Read (RW = 1)  
        movlw   0xFF              ; Set up constant 255
        movwf   timer1            ; for timer loop counter
LCD_is_busy
        bsf     PortB,LCD_e       ; Set E high
	nop
        movf    PortB,w           ; Read PortB into W
        movwf   LCD_read          ; Save W for later testing
	nop
        bcf     PortB,LCD_e       ; Drop E again
        nop                       ; Wait a
	nop                       ;   while
        bsf     PortB,LCD_e       ; Pulse E high (dummy read of lower nibble),
        nop                       ;   wait,
	nop			  ;
        bcf     PortB,LCD_e       ;   and drop E again
        decf    timer1,f          ; Decrement loop counter
        btfsc   STATUS,B_Z        ; Is loop counter down to zero?
        goto    not_busy          ; If yes, return regardless
        btfsc   LCD_read,7        ; Is Busy Flag (RB7) in save byte clear?
        goto    LCD_is_busy       ; If not, it is busy so jump back
not_busy
        return                    ; 
;
; *****************************************************************************
; * Purpose:  Send Command or Data byte to the LCD                            *
; *           Entry point cmnd2LCD:  Send a Command to the LCD                *
; *           Entry Point data2LCD:  Send a Data byte to the LCD              *
; *                                                                           *
; *   Input:  W has the command or data byte to be sent to the LCD.           *
; *                                                                           *
; *  Output:  None                                                            *
; *****************************************************************************
;
cmnd2LCD   ; ****** Entry point ******
        movwf   LCD_char          ; Save byte to write to LCD
        clrf    rs_value          ; Remember to clear RS  (clear rs_value)   
        bcf     PortB,LCD_rs      ; Set RS for Command to LCD
        goto    write2LCD         ; Go to common code
data2LCD   ; ****** Entry point ********
        movwf   LCD_char          ; Save byte to write to LCD
        bsf     rs_value,0        ; Remember to set RS (set bit 0 of rs_value)
        bsf     PortB,LCD_rs      ; Set RS for Data to LCD
write2LCD
        call    busy_check        ; Check to see if LCD is ready for new data
        clrf    PortB             ; Clear all of Port B (inputs and outputs)
        Bank1			  ; Switch to bank 1 for Tristate operation
        movlw   0x01              ; Set up to enable PortB data pins
        movwf   TRISB             ; Pins (RB7..RB1) are back to outputs
        Bank0			  ; Switch to bank 0
        bcf     PortB,LCD_rw      ; Set LCD back to Write mode  (RW = 0)
        bcf     PortB,LCD_rs      ; Guess RS should be clear              
        btfsc   rs_value,0        ; Should RS be clear?  (is bit 0 == 0?) 
        bsf     PortB,LCD_rs      ; No, set RS                            
;
; Transfer Most Significant nibble  (XXXX portion of XXXXYYYY)
;
        movlw   0x0F              ; Set up mask                                 
        andwf   PortB,f           ; Clear old RB7..RB4 
        movf    LCD_char,w        ; Put byte of data into W
        andlw   0xF0              ; Mask to give XXXX0000 in W
        bsf     PortB,LCD_e       ; Pulse the E line high,
	nop
	iorwf   PortB,f           ; Send to RB7..RB4 without changing RB3..RB0
        nop                       ;   wait, 
	nop			  ;
        bcf     PortB,LCD_e       ;   and drop it again
;
; Transfer Least Significant nibble  (YYYY portion of XXXXYYYY)
;
        movlw   0x0F              ; Set up mask                                 
        andwf   PortB,f           ; Clear old RB7..RB4
        swapf   LCD_char,w        ; Move LS nibble of data to MS position in W
        andlw   0xF0              ; Mask to give YYYY0000 in W
        bsf     PortB,LCD_e       ; Pulse the E line high,
	nop
	iorwf   PortB,f           ; Send to RB7..RB4 without changing RB3..RB0
        nop                       ;   wait,
	nop			  ; 
        bcf     PortB,LCD_e       ;   and drop it again
        return

;==========================================================================================
; Message repository
;
; This section contains the status messages and a routine to control one off updates
; preventing display disturbances
;
;==========================================================================================


; Clear display and position cursor at start of message
; Message flags prevent cyclic clearing of display and visual strobe effect

Msg_line2				
	movlw	0x01		  ; Clear the display for new message
	call	cmnd2LCD
	clrf	msgflag		  ; Clear message flags
        movlw   0xC2              ; Point the LCD to first LCD digit location
        call    cmnd2LCD          ; Send starting digit location to LCD
	return

; IF Prog Message

Prog_msg
	btfsc	msgflag,0	  ; If flag zero new message needed, proceed else return
	return
	call	Msg_line2
        movlw   'O'               
        call    data2LCD   
        movlw   'f'               
        call    data2LCD   
        movlw   'f'               
        call    data2LCD         
	movlw   's'               
        call    data2LCD   
        movlw   'e'               
        call    data2LCD
        movlw   't'               
        call    data2LCD
        movlw   ' ' 
        call    data2LCD
        movlw   'P'               
        call    data2LCD   
        movlw   'r'               
        call    data2LCD   
        movlw   'o'               
        call    data2LCD         
	movlw   'g'               
        call    data2LCD   
	bsf	msgflag,0	  ; Message displayed
	return
;
; IF Stored Message

Stored_msg
	btfsc	msgflag,1	  ; If flag zero new message needed, proceed else return
	return
	call	Msg_line2
        movlw   'S'               
        call    data2LCD   
        movlw   'a'               
        call    data2LCD   
        movlw   'v'               
        call    data2LCD         
	movlw   'e'               
        call    data2LCD   
        movlw   'd'               
        call    data2LCD 
	bsf	msgflag,1	  ; Message displayed
	return
;
; Direct Mode
Direct_msg
	btfsc	msgflag,2	  ; If flag zero new message needed, proceed else return
	return
	call	Msg_line2
        movlw   'D'               
        call    data2LCD   
        movlw   'i'               
        call    data2LCD   
        movlw   'r'               
        call    data2LCD         
	movlw   'e'               
        call    data2LCD   
        movlw   'c'               
        call    data2LCD
        movlw   't'               
        call    data2LCD
	bsf	msgflag,2	  ; Message displayed
	return
;
; Offset Sum
Offset_sum_msg
	btfss	msgflag,0	  ; If in IF Programming mode skip Sum/Diff msg
	btfsc	msgflag,3	  ; If flag zero new message needed, proceed else return
	return
	call	Msg_line2
        movlw   'O'               
        call    data2LCD   
        movlw   'f'               
        call    data2LCD   
        movlw   'f'               
        call    data2LCD         
	movlw   's'               
        call    data2LCD   
        movlw   'e'               
        call    data2LCD
        movlw   't'               
        call    data2LCD
        movlw   ' '               
        call    data2LCD         
	movlw   'S'               
        call    data2LCD   
        movlw   'u'               
        call    data2LCD
        movlw   'm'               
        call    data2LCD
        movlw   ' '               
        call    data2LCD
	bsf	msgflag,3	  ; Message displayed
	return
;
; Offset Diff
Offset_diff_msg
	btfss	msgflag,0	  ; If in IF Programming mode skip Sum/Diff msg
	btfsc	msgflag,4	  ; If flag zero new message needed, proceed else return
	return
	call	Msg_line2
        movlw   'O'               
        call    data2LCD   
        movlw   'f'               
        call    data2LCD   
        movlw   'f'               
        call    data2LCD         
	movlw   's'               
        call    data2LCD   
        movlw   'e'               
        call    data2LCD
        movlw   't'               
        call    data2LCD
        movlw   ' '               
        call    data2LCD         
	movlw   'D'               
        call    data2LCD   
        movlw   'i'               
        call    data2LCD
        movlw   'f'               
        call    data2LCD
        movlw   'f'               
        call    data2LCD
	bsf	msgflag,4	  ; Message displayed
	return

Prog_x_msg
	btfsc	msgflag,5	  ; If flag zero new message needed, proceed else return
	return
	call	Msg_line2
        movlw   'M'               
        call    data2LCD   
        movlw   'u'               
        call    data2LCD   
        movlw   'l'               
        call    data2LCD         
	movlw   't'               
        call    data2LCD   
        movlw   ' '               
        call    data2LCD
        movlw   'F'               
        call    data2LCD
        movlw   'a'               
        call    data2LCD         
	movlw   'c'               
        call    data2LCD   
        movlw   't'               
        call    data2LCD
        movlw   'o'               
        call    data2LCD
        movlw   'r'               
        call    data2LCD
	bsf	msgflag,5	  ; Message displayed
	return

Msg_on_msg
	btfsc	msgflag1,0	  ; If flag zero new message needed, proceed else return
	return
	call	Msg_line2
        movlw   'M'               
        call    data2LCD   
        movlw   'e'               
        call    data2LCD   
        movlw   's'               
        call    data2LCD   
        movlw   's'               
        call    data2LCD   
        movlw   'a'               
        call    data2LCD   
        movlw   'g'               
        call    data2LCD   
        movlw   'e'               
        call    data2LCD   
        movlw   ':'               
        call    data2LCD   
        movlw   ' '               
        movlw   'P'               
        call    data2LCD
        movlw   'e'               
        call    data2LCD         
	movlw   'r'               
        call    data2LCD   
        movlw   'm'               
        call    data2LCD
        movlw   ' '  		  ; Blanking byte to make both of these messages same length             
        call    data2LCD
	bsf	msgflag1,0	  ; Message displayed
	return

Msg_off_msg
	btfsc	msgflag1,1	  ; If flag zero new message needed, proceed else return
	return
	call	Msg_line2
        movlw   'M'               
        call    data2LCD   
        movlw   'e'               
        call    data2LCD   
        movlw   's'               
        call    data2LCD   
        movlw   's'               
        call    data2LCD   
        movlw   'a'               
        call    data2LCD   
        movlw   'g'               
        call    data2LCD   
        movlw   'e'               
        call    data2LCD   
        movlw   ':'               
        call    data2LCD   
        movlw   ' '               
        movlw   'T'               
        call    data2LCD
        movlw   'i'               
        call    data2LCD         
	movlw   'm'               
        call    data2LCD   
        movlw   'e'               
        call    data2LCD
        movlw   'd'               
        call    data2LCD
	bsf	msgflag1,1	  ; Message displayed
	return

Null_msg
	btfsc	msgflag,6	  ; If flag zero new message needed, proceed else return
	return
	movlw	0x01		  ; Clear the display
	call	cmnd2LCD
	bsf	msgflag,6	  ; Message
	return

; *****************************************************************************
; *                                                                           *
; * Purpose:  Wait for a specified number of milliseconds.                    *
; *                                                                           *
; *           Entry point wait_128ms:  Wait for 128 msec                      *
; *           Entry point wait_64ms :  Wait for 64 msec                       *
; *           Entry point wait_32ms :  Wait for 32 msec                       *
; *           Entry point wait_16ms :  Wait for 16 msec                       *
; *           Entry point wait_8ms  :  Wait for 8 msec                        *
; *                                                                           *
; *   Input:  None                                                            *
; *                                                                           *
; *  Output:  None                                                            *
; *                                                                           *
; *****************************************************************************
;
wait_128ms  ; ****** Entry point ******    
        movlw   0x80              ; Set up outer loop 
        movwf   timer1            ;   counter to 128
        goto    outer_loop        ; Go to wait loops
wait_64ms  ; ****** Entry point ******     
        movlw   0x40              ; Set up outer loop
        movwf   timer1            ;   counter to 64
        goto    outer_loop        ; Go to wait loops
wait_32ms   ; ****** Entry point ******    
        movlw   0x20              ; Set up outer loop
        movwf   timer1            ;   counter to 32
        goto    outer_loop        ; Go to wait loops
wait_16ms   ; ****** Entry point ******    
        movlw   0x10              ; Set up outer loop
        movwf   timer1            ;   counter to 16  
        goto    outer_loop        ; Go to wait loops
wait_8ms   ; ****** Entry point ******     
        movlw   0x08              ; Set up outer loop
        movwf   timer1            ;   counter to 8
                                  ; Fall through into wait loops
;
; Wait loops used by other wait routines 
;  - 1 microsecond per instruction (with a 4MHz microprocessor crystal)
;  - 1020 instructions per inner loop
;  - (Timer1 * 1020) instructions (1.020 msec) per outer loop
;  - Round off to 1msec per outer loop
;
outer_loop                        
        movlw   0xFF              ; Set up inner loop counter
        movwf   timer2            ; to 255
inner_loop
	nop			  ; Null to pad out to 1 microsecond
	nop			  ; Null to pad out to 1 microsecond
        decfsz  timer2,f          ; Decrement inner loop counter
        goto    inner_loop        ; If inner loop counter not down to zero, 
                                  ; then go back to inner loop again
        decfsz  timer1,f          ; Yes, Decrement outer loop counter
        goto    outer_loop        ; If outer loop counter not down to zero,
                                  ; then go back to outer loop again
        return                    ; Yes, return to caller
;       
; *****************************************************************************

	END


